<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// ==================== CONFIGURAÇÃO ====================
define('EVOLUTION_API_URL', 'https://api.controlmaxerp.com.br');
define('DEBUG_MODE', true);

// ==================== FUNÇÕES ====================
function logDebug($message, $data = null) {
    if (DEBUG_MODE) {
        $logMessage = date('Y-m-d H:i:s') . ' - ' . $message;
        if ($data) {
            $logMessage .= " - " . json_encode($data, JSON_UNESCAPED_UNICODE);
        }
        error_log($logMessage);
    }
}

// Função para obter a API Key - agora é simplesmente o nome da instância
function getApiKeyForInstance($instanceName) {
    return $instanceName;
}

function makeEvolutionRequest($endpoint, $method = 'GET', $data = null, $instanceName = '') {
    $url = EVOLUTION_API_URL . $endpoint;
    
    // A API Key é o próprio nome da instância
    $apiKey = getApiKeyForInstance($instanceName);
    
    logDebug("🔄 Evolution API Request", [
        'endpoint' => $endpoint,
        'method' => $method,
        'url' => $url,
        'instance' => $instanceName,
        'api_key' => $apiKey
    ]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'apikey: ' . $apiKey
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            $jsonData = json_encode($data);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonData);
        }
    } elseif ($method === 'DELETE') {
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    
    curl_close($ch);
    
    logDebug("📥 Evolution API Response", [
        'http_code' => $httpCode,
        'curl_error' => $curlError,
        'response_preview' => substr($response, 0, 200)
    ]);
    
    $decodedResponse = json_decode($response, true);
    
    return [
        'success' => $httpCode >= 200 && $httpCode < 300,
        'data' => $decodedResponse,
        'httpCode' => $httpCode,
        'rawResponse' => $response,
        'curlError' => $curlError
    ];
}

// Função para atualizar configurações da instância
function updateInstanceConfig($instanceName) {
    $config = [
        'disableNotifications' => false,
        'alwaysOnline' => false,
        'readReceipts' => true,
        'syncFullHistory' => false,
        'markOnlineOnConnect' => false,
        'rejectCall' => false,
        'msgRetryCounter' => 3
    ];
    
    logDebug("⚙️ Atualizando configurações da instância", $config);
    
    $result = makeEvolutionRequest('/instance/updateConfig/' . $instanceName, 'POST', $config, $instanceName);
    
    if ($result['success']) {
        logDebug("✅ Configurações atualizadas com sucesso para: " . $instanceName);
        return true;
    } else {
        logDebug("⚠️ Não foi possível atualizar configurações: " . $instanceName, $result);
        return false;
    }
}

// ==================== ROTEAMENTO ====================
$action = $_GET['action'] ?? '';
$instanceName = $_GET['instance'] ?? '';

logDebug("🎯 Nova Requisição", [
    'action' => $action,
    'instance' => $instanceName,
    'method' => $_SERVER['REQUEST_METHOD']
]);

try {
    // Verifica se a instância foi especificada (exceto para health check)
    if ($action !== 'health' && empty($instanceName)) {
        throw new Exception('Nome da instância não especificado');
    }
    
    switch ($action) {
        case 'health':
            echo json_encode([
                'status' => 'ok',
                'timestamp' => date('Y-m-d H:i:s'),
                'message' => 'API Evolution funcionando!'
            ]);
            break;
            
        case 'instances':
            // Para listar instâncias, usa a instância da URL
            $result = makeEvolutionRequest('/instance/fetchInstances', 'GET', null, $instanceName);
            
            if ($result['success']) {
                echo $result['rawResponse'];
            } else {
                http_response_code(500);
                echo json_encode([
                    'error' => 'Erro ao buscar instâncias',
                    'http_code' => $result['httpCode']
                ]);
            }
            break;
            
        case 'connect':
            logDebug("🔗 Conectando instância: " . $instanceName);
            
            // PRIMEIRO: Atualizar configurações para permitir notificações
            logDebug("⚙️ Atualizando configurações para permitir notificações");
            $configUpdated = updateInstanceConfig($instanceName);
            
            if (!$configUpdated) {
                logDebug("⚠️ Continuando conexão mesmo com falha na atualização de configurações");
            }
            
            // SEGUNDO: Tenta conectar diretamente
            $connectResult = makeEvolutionRequest('/instance/connect/' . $instanceName, 'GET', null, $instanceName);
            
            if ($connectResult['success']) {
                logDebug("✅ Conexão bem-sucedida");
                echo $connectResult['rawResponse'];
            } else {
                // Se falhou, tenta criar primeiro e depois conectar
                if ($connectResult['httpCode'] === 404) {
                    logDebug("📝 Instância não existe, criando: " . $instanceName);
                    
                    $createResult = makeEvolutionRequest('/instance/create', 'POST', [
                        'instanceName' => $instanceName
                    ], $instanceName);
                    
                    if ($createResult['success']) {
                        sleep(2);
                        
                        // Atualizar configurações da nova instância
                        updateInstanceConfig($instanceName);
                        
                        // Tenta conectar novamente após criar
                        $connectResult = makeEvolutionRequest('/instance/connect/' . $instanceName, 'GET', null, $instanceName);
                        if ($connectResult['success']) {
                            echo $connectResult['rawResponse'];
                        } else {
                            throw new Exception('Instância criada mas falha ao conectar: ' . 
                                ($connectResult['curlError'] ?? 'Erro desconhecido'));
                        }
                    } else {
                        throw new Exception('Falha ao criar instância: ' . 
                            ($createResult['curlError'] ?? 'Erro desconhecido'));
                    }
                } else {
                    throw new Exception('Falha ao conectar instância: ' . 
                        ($connectResult['curlError'] ?? 'HTTP ' . $connectResult['httpCode']));
                }
            }
            break;
            
        case 'status':
            $result = makeEvolutionRequest('/instance/connectionState/' . $instanceName, 'GET', null, $instanceName);
            
            if ($result['success']) {
                echo $result['rawResponse'];
            } else {
                echo json_encode([
                    'instance' => [
                        'instanceName' => $instanceName,
                        'state' => 'close'
                    ]
                ]);
            }
            break;
            
        case 'logout':
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                throw new Exception('Método não permitido');
            }
            
            $result = makeEvolutionRequest('/instance/logout/' . $instanceName, 'DELETE', null, $instanceName);
            
            if ($result['success']) {
                echo $result['rawResponse'];
            } else {
                throw new Exception('Falha ao desconectar instância');
            }
            break;
            
        case 'config':
            // Endpoint adicional para atualizar configurações manualmente
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                throw new Exception('Método não permitido');
            }
            
            $configUpdated = updateInstanceConfig($instanceName);
            
            if ($configUpdated) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Configurações atualizadas com sucesso',
                    'config' => [
                        'disableNotifications' => false,
                        'alwaysOnline' => false,
                        'readReceipts' => true,
                        'syncFullHistory' => false,
                        'markOnlineOnConnect' => false
                    ]
                ]);
            } else {
                throw new Exception('Falha ao atualizar configurações');
            }
            break;
            
        default:
            http_response_code(400);
            echo json_encode([
                'error' => 'Ação não especificada ou inválida',
                'available_actions' => [
                    'health', 'instances', 'connect', 'status', 'logout', 'config'
                ]
            ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => $e->getMessage()
    ]);
}
?>